/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *  
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *  
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 * 
 */
package com.jimischopp.ant;

import org.apache.tools.ant.BuildException;
import org.apache.tools.ant.Project;
import org.apache.tools.ant.taskdefs.Javac;
import org.apache.tools.ant.taskdefs.compilers.CompilerAdapter;
import org.apache.tools.ant.taskdefs.compilers.CompilerAdapterFactory;

import java.io.File;


/**
 * Compiles Java source files, grouping files into sets of XX (configurable) files, and
 * passing each group separately to the java compiler. This task directly extends the "javac"
 * task, and therefore supports/requires any arguments as specified by javac. Additionally,
 * the FileGroupingJavac task takes the following argument:
 * <ul>
 * <li>groupsize
 * </ul>
 * <p>
 * As with javac, when this task executes, it will recursively scan the sourcedir and
 * destdir looking for Java source files to compile. This task makes its
 * compile decision based on timestamp.
 * <p>
 * However, the files selected for compilation will be additionaly grouped into sets of
 * (not larger than) "groupsize" files. The compiler is then called once for each group (instead
 * of just once for the entire fileset, as with javac.) This is useful when the fileset is large
 * and you typically receive out-of-memory exceptions while compiling. This does NOT take into
 * account dependency issues however, so, if file A must be compiled in the same pass as file B,
 * and file B was put into another group, then the compile for file A may fail. (However, if you
 * do incremental builds - which is most likely the case for large filesets - then file A will
 * compile the second time since file B would have already been compiled.)
 * <p>
 * When creating the filegroups, FileGroupingJavac evenly distributes the files into the groups. For
 * example, if you specify groupsize=100 and the initial fileset contains 102 files, then
 * FileGroupingJavac will make 2 groups of 51 files (NOT 1 group of 100 files, and
 * another of just 2).
 * 
 * <p>
 * Copyright 2003, James Schopp
 *
 * @author James Schopp
 * @see javac
 * @since Oct 27, 2003
 * @ant.task category="java"
 * 
 */
public class FileGroupingJavac
    extends Javac
{
    //~ Static fields/initializers /////////////////////////////////////////////////////////////////

    public static final int     DEFAULT_GROUP_SIZE = 100;
    private static final String FAIL_MSG = "Compile failed; see the compiler error output for details.";

    //~ Instance fields ////////////////////////////////////////////////////////////////////////////

    private int m_iGroupSize = DEFAULT_GROUP_SIZE;

    //~ Methods ////////////////////////////////////////////////////////////////////////////////////


    /**
     * The setter for the "groupsize" attribute
     * @param newSize
     */
    public void setGroupsize(int newSize)
    {
        m_iGroupSize = newSize;
    }

    /**
     * The getter for the "GroupSize" attribute
     * @return the value of the "GroupSize" attribute
     */
    public int getGroupsize()
    {
        return m_iGroupSize;
    }


    /**
     * Perform the compilation, regrouping files as necessary.
     * If no groups are necessary, then the "original" javac is called directly.
     */
    protected void compile()
    {
        //get how many groups we'll need...
        int numGroups = compileList.length / m_iGroupSize;
        numGroups += ((compileList.length % m_iGroupSize) == 0) ? 0 : 1;

        //see if we need to do any splitting or not...
        if(numGroups > 1) {
            log(compileList.length + " total files. Dividing files into " + numGroups
                + " groups of maximum " + m_iGroupSize + " files."
               );
        } else {
            super.compile();

            return;
        }

        String          compilerImpl    = getCompiler();
        CompilerAdapter adapter         = CompilerAdapterFactory.getCompiler(compilerImpl, this);
        File            destDir         = getDestdir();
        File[]          compileListOrig = compileList;

        int             currGroup       = 0;
        int             currGroupSize   = 0;
        int             currFile        = 0;

        for(currGroup = 0, currFile = 0;
            currGroup < numGroups;
            currGroup++, currFile += currGroupSize
           ) {
            //how many are in the current group? (only last case is different)            
            if(currGroup == numGroups - 1) {
                currGroupSize = (compileListOrig.length / numGroups)
                                + (compileListOrig.length % numGroups);
            } else {
                currGroupSize = (compileListOrig.length / numGroups);
            }

            //get a new array if necessary...
            if(compileList == null || compileList.length != currGroupSize) {
                compileList = new File[currGroupSize];
            }

            //copy the current set of files over to the temp array...
            System.arraycopy(compileListOrig, currFile, compileList, 0, currGroupSize);

            log("Compiling source files " + (currFile + 1) + " to " + (currFile + currGroupSize)
                + (destDir != null ? " to " + destDir : "")
               );

            //see if we are supposed to list out the files...
            if(listFiles) {
                for(int i = 0; i < currGroupSize; i++) {
                    String filename = compileList[i].getAbsolutePath();
                    log(filename);
                }
            }

            // now we need to populate the compiler adapter with the new file set...
            adapter.setJavac(this);

            // finally, lets execute the compiler!!
            if(!adapter.execute()) {
                if(failOnError) {
                    throw new BuildException(FAIL_MSG, getLocation());
                } else {
                    log(FAIL_MSG, Project.MSG_ERR);
                }
            }
        }

        //copy the real list back over...
        compileList = compileListOrig;
    }
}
