/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *  
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *  
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 * 
 */
package com.jimischopp.jdbcstatwrapper;

import java.io.PrintStream;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.SortedMap;

//import com.jimischopp.util.DateAndTimeUtil;

/**
 * 
 * A JDBC Tracing utility. This utility allows you to "wrap" all JDBC database activity,
 * starting with the database driver, with an end to
 * <ul>
 * <li>view unclosed statements
 * <li>view unclosed resultsets
 * <li>view unclosed connections
 * <li>view unused statements
 * <li>view duplicate queries/prepared statements (checks for parameters too)
 * </ul>
 * The trace even prints out a stack trace of opened statements that were not
 * closed, so you can see exactly where it gets opened.
 * 
 * <p>
 * This tool is most useful when you use unit-test based development, so that after
 * each run of a service, you can log all the database activity that the service used,
 * and then make the correct adjustments to your code for best performance.
 * 
 * <p>
 * To use it, all you need to do is: prepend "jdbc:statJdbcDriver:" to the beginning
 * of your normal JDBC URL, and ensure that the StatDriver class is also loaded. For
 * example, a typical scenario would be,
 * <p>
 * BEFORE:
 * <pre>
 * 	Class.forName("oracle.jdbc.driver.OracleDriver");
 *  Connection conn = DriverManager.getConnection("jdbc:oracle:thin:@//myhost:1521/orcl", "scott", "tiger"); 
 * </pre> 
 * AFTER:
 * <pre>
 *  Class.forName("oracle.jdbc.driver.OracleDriver");
 *  Class.forName("com.jimischopp.jdbcstatwrapper.StatDriver");
 *  Connection conn = DriverManager.getConnection("jdbc:statJdbcDriver:oracle:thin:@//myhost:1521/orcl", "scott", "tiger");
 * </pre>
 * 
 * Then when you wish to see the stats, simply call:
 * <pre>
 *  JDBCStatWrapperMonitor.logStats();
 * </pre>
 * 
 * <p>
 * Copyright 2007, James Schopp
 *
 * @author James Schopp
 */
public class JDBCStatWrapperMonitor {
    
    protected static Map queryStats = new HashMap(); //queryString==>QueryStats
    protected static QueryStats totalStats = new QueryStats();
    protected static long connectionsOpened = 0;
    protected static long connectionsClosed = 0;
    
    protected static Map objectsCreated = new HashMap();
    protected static Map objectsClosedStacks = new HashMap(); //objectId=>List(stacks)
    protected static Set objectsNotClosed = new HashSet();
    protected static Set objectsClosed = new HashSet();    
    protected static Set objectsClosedMultipleTimes = new HashSet();
        
    protected static Set statementsExecuted = new HashSet();
    protected static Set statementsNotExecuted = new HashSet();
    protected static Set statementsExecutedMultipleTimes = new HashSet();
    
    private final static String PARAM_KEY_SEP = ","; 
    private final static String PARAM_KEY_NULL = "<null params>";
    /* private */ static String makeKeyFromParamMap(SortedMap params)
    {
        if (params == null || params.size()==0) {
            return PARAM_KEY_NULL;
        }
        StringBuffer buff = new StringBuffer(100);
        for (Iterator it=params.keySet().iterator(); it.hasNext();) {
            buff.append(params.get(it.next()).toString()).append(PARAM_KEY_SEP);
        }
        buff.delete(buff.length() - PARAM_KEY_SEP.length(), buff.length());
        return buff.toString();        
    }
    
    public static long getConnectionsClosed() {
        return connectionsClosed;
    }
    public static void setConnectionsClosed(long connectionsClosed) {
        JDBCStatWrapperMonitor.connectionsClosed = connectionsClosed;
    }
    public static long getConnectionsOpened() {
        return connectionsOpened;
    }
    public static void setConnectionsOpened(long connectionsOpened) {
        JDBCStatWrapperMonitor.connectionsOpened = connectionsOpened;
    }
        
    public static void incrementTimesExecuted(String query, SortedMap params, long execTime, int objectId)
    {
        if (query != null) {
            QueryStats stats = (QueryStats) queryStats.get(query);
            if (stats == null) {
                stats = new QueryStats();
                stats.setQuery(query);
                queryStats.put(query, stats);            
            }
            
            stats.getParams().add(makeKeyFromParamMap(params));
            
            stats.timesExecuted++;
            stats.totalExecutionTime += execTime;
        }
        
        totalStats.timesExecuted++;
        totalStats.totalExecutionTime += execTime;
        
        executeStatement(objectId);
    }
    public static void incrementResultSetsOpened(String query, SortedMap params, int objectId)
    {      
        if (query != null) {
            QueryStats stats = (QueryStats) queryStats.get(query);
            if (stats == null) {
                stats = new QueryStats();
                stats.setQuery(query);
                queryStats.put(query, stats);            
            }
            
            stats.getParams().add(makeKeyFromParamMap(params));
            
            stats.resultSetsOpened++;
        }
        
        totalStats.resultSetsOpened++;   
        openObject(objectId);
    }
    public static void incrementResultSetsClosed(String query, SortedMap params, int objectId)
    {        
        if (query != null) {
            QueryStats stats = (QueryStats) queryStats.get(query);
            if (stats == null) {
                stats = new QueryStats();
                stats.setQuery(query);
                queryStats.put(query, stats);            
            }
            
            stats.getParams().add(makeKeyFromParamMap(params));
            
            stats.resultSetsClosed++;
        }
                
        totalStats.resultSetsClosed++;   
        closeObject(objectId);
    }
    public static void incrementStatementsOpened(String query, int objectId)
    {        
        if (query != null) {
            QueryStats stats = (QueryStats) queryStats.get(query);
            if (stats == null) {
                stats = new QueryStats();
                stats.setQuery(query);
                queryStats.put(query, stats);            
            }
                    
            stats.statementsOpened++;
        }
                
        totalStats.statementsOpened++;    
        openObject(objectId);
        statementsNotExecuted.add(new Integer(objectId));
    }
    public static void incrementStatementsClosed(String query, int objectId)
    {        
        if (query != null) {
            QueryStats stats = (QueryStats) queryStats.get(query);
            if (stats == null) {
                stats = new QueryStats();
                stats.setQuery(query);
                queryStats.put(query, stats);            
            }
                    
            stats.statementsClosed++;
        }
                
        totalStats.statementsClosed++;     
        closeObject(objectId);
    } 
    public static void incrementConnectionsOpened(int objectId)
    {
        connectionsOpened++;      
        openObject(objectId);
    }
    public static void incrementConnectionsClosed(int objectId)
    {
        connectionsClosed++;
        closeObject(objectId);        
    }
    
    private static void openObject(int objectId)
    {
        Integer i = new Integer(objectId);
        StackTraceElement[] stack = getUnwrappedStackTrace();        
        objectsCreated.put(i, stack);
        objectsNotClosed.add(i);        
    }
    private static void closeObject(int objectId)
    {
        Integer i = new Integer(objectId);
        objectsNotClosed.remove(i);
        if (objectsClosed.contains(i)) {
            objectsClosedMultipleTimes.add(i);
        } else {
            objectsClosed.add(i);
        }

        //now keep track of where it was closed...
        List closingStacks = (List) objectsClosedStacks.get(i);
        if (closingStacks == null) {
            closingStacks = new ArrayList();
            objectsClosedStacks.put(i, closingStacks);
        }
        closingStacks.add(getUnwrappedStackTrace());
    }
    private static void executeStatement(int objectId)
    {
        Integer i = new Integer(objectId);
        
        statementsNotExecuted.remove(i);
        if (statementsExecuted.contains(i)) {
            statementsExecutedMultipleTimes.add(i);
        } else {
            statementsExecuted.add(i);
        }
    }

    private static boolean isStackTraceElementPartOfThisPackage(StackTraceElement ste)
    {
        if (ste.getClassName().startsWith(JDBCStatWrapperMonitor.class.getPackage().getName())) {
            return true;
        }
        return false;
    }
    private static StackTraceElement[] getUnwrappedStackTrace()
    {
        StackTraceElement[] stack = null;
        try {
            throw new Exception();
        } catch (Exception e) {
            e.fillInStackTrace();
            stack = e.getStackTrace();
        }
        
        //see if we can get rid of the top of the stack (this method right here, right now!!)
        if (stack != null && stack.length > 0) {
            int stackIndx=0;
            while (stackIndx<stack.length && isStackTraceElementPartOfThisPackage(stack[stackIndx])) {
                stackIndx++;
            }
            if (stackIndx < stack.length) {
                StackTraceElement[] stackShort = new StackTraceElement[stack.length-stackIndx];
                System.arraycopy(stack, stackIndx, stackShort, 0, stack.length-stackIndx);
                stack = stackShort;
            }
        }
        return stack;        
    }
    
    private static String getFormattedPercent(double percentRaw, int precision)
    {
        precision = Math.abs(precision);
        
        //BigDecimal bd = new BigDecimal(percentRaw * 100D);
        //bd = bd.setScale(precision, BigDecimal.ROUND_HALF_UP);
        //return bd.toString() + "%";
        
        double percentLeftD = Math.floor(percentRaw * 100D);        
        long percentRight = (long)((percentRaw - (percentLeftD / 100D)) * (1000D * (double)precision));        
        return (long)percentLeftD + (precision>=1 ? "." + percentRight : "" ) + "%";        
    }
    
    /**
     * Internal class for maintaining the statistics.
     *  
     * <p>
     * Copyright 2007, James Schopp
     * </p>
     *
     * @see com.jimischopp.jdbcstatwrapper.JDBCStatWrapperMonitor
     * @author James Schopp
     * @since Jan 31, 2007
     *
     */
    public static class QueryStats {
        protected String query;
        protected Set uniqueParams = new HashSet(); //params are 
        protected long resultSetsOpened;
        protected long resultSetsClosed;
        protected long statementsOpened;
        protected long statementsClosed;    
        protected long timesExecuted;        
        protected long totalExecutionTime;
        
        public long getAverageExecutionTime()
        {
            try {
                return totalExecutionTime / timesExecuted;
            } catch (Exception e) {}
            
            return 0;
        }
        
        public Set getParams() {
            return uniqueParams;
        }
        public void setParams(Set uniqueParams) {
            this.uniqueParams = uniqueParams;
        }
        public String getQuery() {
            return query;
        }
        public void setQuery(String query) {
            this.query = query;
        }
        public long getResultSetsClosed() {
            return resultSetsClosed;
        }
        public void setResultSetsClosed(long resultSetsClosed) {
            this.resultSetsClosed = resultSetsClosed;
        }
        public long getResultSetsOpened() {
            return resultSetsOpened;
        }
        public void setResultSetsOpened(long resultSetsOpened) {
            this.resultSetsOpened = resultSetsOpened;
        }
        public long getStatementsClosed() {
            return statementsClosed;
        }
        public void setStatementsClosed(long statementsClosed) {
            this.statementsClosed = statementsClosed;
        }
        public long getStatementsOpened() {
            return statementsOpened;
        }
        public void setStatementsOpened(long statementsOpened) {
            this.statementsOpened = statementsOpened;
        }
        public long getTotalExecutionTime() {
            return totalExecutionTime;
        }
        public void setTotalExecutionTime(long totalExecutionTime) {
            this.totalExecutionTime = totalExecutionTime;
        }

        public long getTimesExecuted() {
            return timesExecuted;
        }

        public void setTimesExecuted(long timesExecuted) {
            this.timesExecuted = timesExecuted;
        }

        public double getPercentExecutionTime() {
            return (double) getTotalExecutionTime() / (double)totalStats.getTotalExecutionTime();
        }                
    }
    
  
    public static void logStats()
    {
        PrintStream ps = System.out;
        ps.println("Total JDBC Stats");
        ps.println("==========");
        ps.println("Total Connection Opened: " + getConnectionsOpened());
        ps.println("Total Connection Closed: " + getConnectionsClosed());
        ps.println("Total Statements Opened: " + totalStats.getStatementsOpened());
        ps.println("Total Statements Closed: " + totalStats.getStatementsClosed());
        ps.println("Total Queries Executed : " + totalStats.getTimesExecuted());
        ps.println("Total Exec Time        : " + totalStats.getTotalExecutionTime()); //DateAndTimeUtil.getFormattedElapsedTime(totalStats.getTotalExecutionTime()) + " (" + totalStats.getTotalExecutionTime() + ")");        
        ps.println("Total Avg Exec Time    : " + totalStats.getAverageExecutionTime()); //DateAndTimeUtil.getFormattedElapsedTime(totalStats.getAverageExecutionTime()) + " (" + totalStats.getAverageExecutionTime() + ")");
        ps.println("Total ResultSets Opened: " + totalStats.getResultSetsOpened());
        ps.println("Total ResultSets Closed: " + totalStats.getResultSetsClosed());        
        
        
        ps.println("=========="); 
        ps.println("StackTrace of unclosed objects:");
        int x = 0;
        for (Iterator it=objectsNotClosed.iterator(); it.hasNext();) {
            ps.println("Object " + (++x));        
            StackTraceElement[] stack = (StackTraceElement[]) objectsCreated.get(it.next());
            if (stack==null) continue;
            for (int i=0; i<stack.length; i++) {
                ps.println("\t\t" + stack[i].toString());
            }
        }
        
        
        
        
        ps.println("=========="); 
        ps.println("StackTrace of objects closed multiple times:");
        x = 0;
        for (Iterator it=objectsClosedMultipleTimes.iterator(); it.hasNext();) {
            Integer objectId = (Integer) it.next();
            ps.println("Object " + (++x));
            StackTraceElement[] stack = (StackTraceElement[]) objectsCreated.get(objectId);
            for (int i=0; i<stack.length; i++) {
                ps.println("\t\t" + stack[i].toString());
            }
            ps.println("\t\t==========");
            ps.println("\t\tStackTrace of each closing:");
            
            List closings = (List) objectsClosedStacks.get(objectId);
            for(int y = 0; y<closings.size(); y++) {
                ps.println("\t\tClosing " + y);        
                StackTraceElement[] stackClosing = (StackTraceElement[]) closings.get(y);                
                for (int i=0; i<stackClosing.length; i++) {
                    ps.println("\t\t\t" + stackClosing[i].toString());
                }                
            }
        }
        
        
        /*
        ps.println("=========="); 
        ps.println("StackTrace of statements opened but not executed:");
        x = 0;
        for (Iterator it=statementsNotExecuted.iterator(); it.hasNext();) {
            ps.println("Object " + (++x));        
            StackTraceElement[] stack = (StackTraceElement[]) objectsCreated.get(it.next());
            if (stack==null) continue;
            for (int i=0; i<stack.length; i++) {
                ps.println("\t\t" + stack[i].toString());
            }
        }
        */
        
        ps.println("=========="); 
        ps.println("Per-Query JDBC Stats");
        for (Iterator it=queryStats.values().iterator(); it.hasNext();) {
            QueryStats perQueryStats = (QueryStats) it.next();
            
            ps.println("\t==========");            
            ps.println("\tQuery: " + perQueryStats.getQuery());
            ps.println("\t\tStatements Opened: " + perQueryStats.getStatementsOpened());
            ps.println("\t\tStatements Closed: " + perQueryStats.getStatementsClosed() + (perQueryStats.getStatementsClosed() < perQueryStats.getStatementsOpened() ? "\t\t\tSTATEMENTS NOT CLOSED" : ""));
            ps.println("\t\tQueries Executed : " + perQueryStats.getTimesExecuted() + (perQueryStats.getStatementsOpened() > perQueryStats.getTimesExecuted() ? "\t\t\tUNUSED STATEMENTS OPENED" : ""));
            ps.println("\t\tTotal Exec Time  : " + perQueryStats.getTotalExecutionTime());//DateAndTimeUtil.getFormattedElapsedTime(perQueryStats.getTotalExecutionTime()));
            ps.println("\t\tAvg Exec Time    : " + perQueryStats.getAverageExecutionTime());//DateAndTimeUtil.getFormattedElapsedTime(perQueryStats.getAverageExecutionTime()));
            ps.println("\t\tPercent of Total : " + getFormattedPercent(perQueryStats.getPercentExecutionTime(), 1));
            ps.println("\t\tResultSets Opened: " + perQueryStats.getResultSetsOpened());
            ps.println("\t\tResultSets Closed: " + perQueryStats.getResultSetsClosed() + (perQueryStats.getResultSetsClosed() < perQueryStats.getResultSetsOpened() ? "\t\t\tRESULTSETS NOT CLOSED" : ""));
            ps.println("\t\tUniqueParams Used: " + perQueryStats.getParams().size() + (perQueryStats.getParams().size()!=0 && perQueryStats.getParams().size()<perQueryStats.getTimesExecuted() ? "\t\t\tDUPLICATE QUERIES ISSUED" : ""));
            /*
            ps.print("\t\t\t");
            for (Iterator itParams=perQueryStats.getParams().iterator(); itParams.hasNext();) {
                String paramList = itParams.next().toString();                
                ps.println("\t\t\t(" + paramList + ")");
            }            
            ps.println("");
            */
        }        
    }    
}
