/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *  
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *  
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 * 
 */
package com.jimischopp.jdbcstatwrapper;

import java.io.InputStream;
import java.io.Reader;
import java.math.BigDecimal;
import java.net.URL;
import java.sql.Array;
import java.sql.Blob;
import java.sql.Clob;
import java.sql.Date;
import java.sql.ParameterMetaData;
import java.sql.PreparedStatement;
import java.sql.Ref;
import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.SQLException;
import java.sql.Time;
import java.sql.Timestamp;
import java.util.Calendar;
import java.util.Iterator;
import java.util.SortedMap;
import java.util.TreeMap;

/**
 * PreparedStatement wrapper for the JDBCStatWrapper JDBC tracing utility.
 *  
 * <p>
 * Copyright 2007, James Schopp
 * </p>
 * 
 * @see com.jimischopp.jdbcstatwrapper.JDBCStatWrapperMonitor
 * @author James Schopp
 * @since Jan 31, 2007
 *
 */
public class StatPreparedStatement extends StatStatement implements PreparedStatement  {
    
    protected final static String BATCHPARAM_PREFIX = "Batch_";
    protected int batchCount=0;
        
    public StatPreparedStatement(PreparedStatement delegate, String query) {
        super(delegate, query);
    }

    public void addBatch() throws SQLException {
        ((PreparedStatement)delegate).addBatch();
        
        SortedMap mapCurrentBatch = new TreeMap();
        
        for (Iterator it=this.params.keySet().iterator(); it.hasNext();) {
            Object i = it.next();
            if (i instanceof Iterator) {
                mapCurrentBatch.put(i, params.get(i));
                params.remove(i);
            }
        }
        
        String consolidateParamsThisBatch = JDBCStatWrapperMonitor.makeKeyFromParamMap(mapCurrentBatch);
        params.put(BATCHPARAM_PREFIX + (++batchCount), consolidateParamsThisBatch);        
    }
    
    public void clearParameters() throws SQLException {
        ((PreparedStatement)delegate).clearParameters();
    }
    
    public boolean execute() throws SQLException {
        
        long start = System.currentTimeMillis();
        
        boolean ret = ((PreparedStatement)delegate).execute();
        
        long stop = System.currentTimeMillis();
        JDBCStatWrapperMonitor.incrementTimesExecuted(query, params, stop - start, delegate.hashCode());
        
        return ret;
    }

    public ResultSet executeQuery() throws SQLException {
        long start = System.currentTimeMillis();
        
        ResultSet ret = ((PreparedStatement)delegate).executeQuery();
        
        long stop = System.currentTimeMillis();
        JDBCStatWrapperMonitor.incrementTimesExecuted(query, params, stop - start, delegate.hashCode());
        
        return new StatResultSet(ret, query, params);
    }

    public int executeUpdate() throws SQLException {
        long start = System.currentTimeMillis();
        
        int ret =  ((PreparedStatement)delegate).executeUpdate();
        
        long stop = System.currentTimeMillis();
        JDBCStatWrapperMonitor.incrementTimesExecuted(query, params, stop - start, delegate.hashCode());
        
        return ret;
    }

    public ResultSetMetaData getMetaData() throws SQLException {
        return ((PreparedStatement)delegate).getMetaData();
    }

    public ParameterMetaData getParameterMetaData() throws SQLException {
        return ((PreparedStatement)delegate).getParameterMetaData();
    }

    public void setArray(int i, Array x) throws SQLException {
        ((PreparedStatement)delegate).setArray(i, x);
        
        params.put(new Integer(i), "array");
    }

    public void setAsciiStream(int parameterIndex, InputStream x, int length) throws SQLException {
        ((PreparedStatement)delegate).setAsciiStream(parameterIndex, x, length);
        
        params.put(new Integer(parameterIndex), "ascii stream");
    }

    public void setBigDecimal(int parameterIndex, BigDecimal x) throws SQLException {
        ((PreparedStatement)delegate).setBigDecimal(parameterIndex, x);
        
        params.put(new Integer(parameterIndex), x);
    }

    public void setBinaryStream(int parameterIndex, InputStream x, int length) throws SQLException {
        ((PreparedStatement)delegate).setBinaryStream(parameterIndex, x, length);
        
        params.put(new Integer(parameterIndex), "input stream");
    }

    public void setBlob(int i, Blob x) throws SQLException {
        ((PreparedStatement)delegate).setBlob(i, x);
        
        params.put(new Integer(i), "blob");
    }

    public void setBoolean(int parameterIndex, boolean x) throws SQLException {
        ((PreparedStatement)delegate).setBoolean(parameterIndex, x);
        
        params.put(new Integer(parameterIndex), new Boolean(x));
    }

    public void setByte(int parameterIndex, byte x) throws SQLException {
        ((PreparedStatement)delegate).setByte(parameterIndex, x);
        
        params.put(new Integer(parameterIndex), new Byte(x));
    }

    public void setBytes(int parameterIndex, byte[] x) throws SQLException {
        ((PreparedStatement)delegate).setBytes(parameterIndex, x);
        
        params.put(new Integer(parameterIndex), "byte array");
    }

    public void setCharacterStream(int parameterIndex, Reader reader, int length) throws SQLException {
        ((PreparedStatement)delegate).setCharacterStream(parameterIndex, reader, length);
        
        params.put(new Integer(parameterIndex), "character stream");
    }

    public void setClob(int i, Clob x) throws SQLException {
        ((PreparedStatement)delegate).setClob(i, x);
        
        params.put(new Integer(i), "clob");
    }

    public void setDate(int parameterIndex, Date x, Calendar cal) throws SQLException {
        ((PreparedStatement)delegate).setDate(parameterIndex, x, cal);
        
        params.put(new Integer(parameterIndex), x);
    }

    public void setDate(int parameterIndex, Date x) throws SQLException {
        ((PreparedStatement)delegate).setDate(parameterIndex, x);
        
        params.put(new Integer(parameterIndex), x);
    }

    public void setDouble(int parameterIndex, double x) throws SQLException {
        ((PreparedStatement)delegate).setDouble(parameterIndex, x);
        
        params.put(new Integer(parameterIndex), new Double(x));
    }

    public void setFloat(int parameterIndex, float x) throws SQLException {
        ((PreparedStatement)delegate).setFloat(parameterIndex, x);
        
        params.put(new Integer(parameterIndex), new Float(x));
    }

    public void setInt(int parameterIndex, int x) throws SQLException {
        ((PreparedStatement)delegate).setInt(parameterIndex, x);
        
        params.put(new Integer(parameterIndex), new Integer(x));
    }

    public void setLong(int parameterIndex, long x) throws SQLException {
        ((PreparedStatement)delegate).setLong(parameterIndex, x);
        
        params.put(new Integer(parameterIndex), new Long(x));
    }

    public void setNull(int paramIndex, int sqlType, String typeName) throws SQLException {
        ((PreparedStatement)delegate).setNull(paramIndex, sqlType, typeName);
        
        params.put(new Integer(paramIndex), "null");
    }

    public void setNull(int parameterIndex, int sqlType) throws SQLException {
        ((PreparedStatement)delegate).setNull(parameterIndex, sqlType);
        
        params.put(new Integer(parameterIndex), "null");
    }

    public void setObject(int parameterIndex, Object x, int targetSqlType, int scale) throws SQLException {
        ((PreparedStatement)delegate).setObject(parameterIndex, x, targetSqlType, scale);
        
        params.put(new Integer(parameterIndex), x);
    }

    public void setObject(int parameterIndex, Object x, int targetSqlType) throws SQLException {
        ((PreparedStatement)delegate).setObject(parameterIndex, x, targetSqlType);
        
        params.put(new Integer(parameterIndex), x);
    }

    public void setObject(int parameterIndex, Object x) throws SQLException {
        ((PreparedStatement)delegate).setObject(parameterIndex, x);
        
        params.put(new Integer(parameterIndex), x);
    }

    public void setRef(int i, Ref x) throws SQLException {
        ((PreparedStatement)delegate).setRef(i, x);
        
        params.put(new Integer(i), "ref");
    }

    public void setShort(int parameterIndex, short x) throws SQLException {
        ((PreparedStatement)delegate).setShort(parameterIndex, x);
        
        params.put(new Integer(parameterIndex), new Short(x));
    }

    public void setString(int parameterIndex, String x) throws SQLException {
        ((PreparedStatement)delegate).setString(parameterIndex, x);
        
        params.put(new Integer(parameterIndex), x);
    }

    public void setTime(int parameterIndex, Time x, Calendar cal) throws SQLException {
        ((PreparedStatement)delegate).setTime(parameterIndex, x, cal);
        
        params.put(new Integer(parameterIndex), x);
    }

    public void setTime(int parameterIndex, Time x) throws SQLException {
        ((PreparedStatement)delegate).setTime(parameterIndex, x);
        
        params.put(new Integer(parameterIndex), x);
    }

    public void setTimestamp(int parameterIndex, Timestamp x, Calendar cal) throws SQLException {
        ((PreparedStatement)delegate).setTimestamp(parameterIndex, x, cal);
        
        params.put(new Integer(parameterIndex), x);
    }

    public void setTimestamp(int parameterIndex, Timestamp x) throws SQLException {
        ((PreparedStatement)delegate).setTimestamp(parameterIndex, x);
        
        params.put(new Integer(parameterIndex), x);
    }

    public void setUnicodeStream(int parameterIndex, InputStream x, int length) throws SQLException {
        ((PreparedStatement)delegate).setUnicodeStream(parameterIndex, x, length);
        
        params.put(new Integer(parameterIndex), "input stream");
    }

    public void setURL(int parameterIndex, URL x) throws SQLException {
        ((PreparedStatement)delegate).setURL(parameterIndex, x);
        
        params.put(new Integer(parameterIndex), x);
    }    
}
