/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *  
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *  
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 * 
 */
package com.jimischopp.jdbcstatwrapper;

import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.SQLWarning;
import java.sql.Statement;
import java.util.SortedMap;
import java.util.TreeMap;

/**
 * Statement wrapper for the JDBCStatWrapper JDBC tracing utility.
 *  
 * <p>
 * Copyright 2007, James Schopp
 * </p>
 *
 * @see com.jimischopp.jdbcstatwrapper.JDBCStatWrapperMonitor
 * @author James Schopp
 * @since Jan 31, 2007
 *
 */
public class StatStatement implements Statement {
    
    protected Statement delegate;
    protected String query;
    protected SortedMap params = new TreeMap();
        
    public StatStatement(Statement delegate) {
        this(delegate, null);        
    } 
    public StatStatement(Statement delegate, String sql) {
        super();
        this.delegate = delegate;
        this.query = sql;
        
        JDBCStatWrapperMonitor.incrementStatementsOpened(sql, delegate.hashCode());
    }

    public Statement getDelegate() {
        return delegate;
    }
    
    public void addBatch(String sql) throws SQLException {
        if (!sql.equals(query)) {
            query = sql;
            JDBCStatWrapperMonitor.incrementStatementsOpened(query, delegate.hashCode());
        }
        delegate.addBatch(sql);        
    }

    public void cancel() throws SQLException {
        delegate.cancel();
    }

    public void clearBatch() throws SQLException {
        delegate.clearBatch();
    }

    public void clearWarnings() throws SQLException {
        delegate.clearWarnings();
    }

    public void close() throws SQLException {
        JDBCStatWrapperMonitor.incrementStatementsClosed(query, delegate.hashCode());        
        delegate.close();        
    }

    public boolean execute(String sql, int autoGeneratedKeys) throws SQLException {
        if (!sql.equals(query)) {
            query = sql;
            JDBCStatWrapperMonitor.incrementStatementsOpened(query, delegate.hashCode());
        }
        long start = System.currentTimeMillis();
        
        boolean ret = delegate.execute(sql, autoGeneratedKeys);
        
        long stop = System.currentTimeMillis();
        JDBCStatWrapperMonitor.incrementTimesExecuted(query, params, stop - start, delegate.hashCode());
        
        return ret;
    }

    public boolean execute(String sql, int[] columnIndexes) throws SQLException {
        if (!sql.equals(query)) {
            query = sql;
            JDBCStatWrapperMonitor.incrementStatementsOpened(query, delegate.hashCode());
        }
        long start = System.currentTimeMillis();
        
        boolean ret = delegate.execute(sql, columnIndexes);
        
        long stop = System.currentTimeMillis();
        JDBCStatWrapperMonitor.incrementTimesExecuted(query, params, stop - start, delegate.hashCode());
        
        return ret;
    }

    public boolean execute(String sql, String[] columnNames) throws SQLException {
        if (!sql.equals(query)) {
            query = sql;
            JDBCStatWrapperMonitor.incrementStatementsOpened(query, delegate.hashCode());
        }
        long start = System.currentTimeMillis();
        
        boolean ret =  delegate.execute(sql, columnNames);
        
        long stop = System.currentTimeMillis();
        JDBCStatWrapperMonitor.incrementTimesExecuted(query, params, stop - start, delegate.hashCode());
        
        return ret;
    }

    public boolean execute(String sql) throws SQLException {
        if (!sql.equals(query)) {
            query = sql;
            JDBCStatWrapperMonitor.incrementStatementsOpened(query, delegate.hashCode());
        }
        long start = System.currentTimeMillis();
        
        boolean ret =  delegate.execute(sql);
        
        long stop = System.currentTimeMillis();
        JDBCStatWrapperMonitor.incrementTimesExecuted(query, params, stop - start, delegate.hashCode());
        
        return ret;
    }

    public int[] executeBatch() throws SQLException {
        
        long start = System.currentTimeMillis();
        
        int[] ret =  delegate.executeBatch();
        
        long stop = System.currentTimeMillis();
        JDBCStatWrapperMonitor.incrementTimesExecuted(query, params, stop - start, delegate.hashCode());
        
        return ret;
    }

    public ResultSet executeQuery(String sql) throws SQLException {
        if (!sql.equals(query)) {
            query = sql;
            JDBCStatWrapperMonitor.incrementStatementsOpened(query, delegate.hashCode());
        }
        long start = System.currentTimeMillis();
        
        ResultSet ret = delegate.executeQuery(sql);
        
        long stop = System.currentTimeMillis();
        JDBCStatWrapperMonitor.incrementTimesExecuted(query, params, stop - start, delegate.hashCode());
        
        return new StatResultSet(ret, query, params);
    }

    public int executeUpdate(String sql, int autoGeneratedKeys) throws SQLException {
        if (!sql.equals(query)) {
            query = sql;
            JDBCStatWrapperMonitor.incrementStatementsOpened(query, delegate.hashCode());
        }
        long start = System.currentTimeMillis();
        
        int ret = delegate.executeUpdate(sql, autoGeneratedKeys);
        
        long stop = System.currentTimeMillis();
        JDBCStatWrapperMonitor.incrementTimesExecuted(query, params, stop - start, delegate.hashCode());
        
        return ret;
    }

    public int executeUpdate(String sql, int[] columnIndexes) throws SQLException {
        if (!sql.equals(query)) {
            query = sql;
            JDBCStatWrapperMonitor.incrementStatementsOpened(query, delegate.hashCode());
        }
        long start = System.currentTimeMillis();
        
        int ret =  delegate.executeUpdate(sql, columnIndexes);
        
        long stop = System.currentTimeMillis();
        JDBCStatWrapperMonitor.incrementTimesExecuted(query, params, stop - start, delegate.hashCode());
        
        return ret;
    }

    public int executeUpdate(String sql, String[] columnNames) throws SQLException {
        if (!sql.equals(query)) {
            query = sql;
            JDBCStatWrapperMonitor.incrementStatementsOpened(query, delegate.hashCode());
        }
        long start = System.currentTimeMillis();
        
        int ret =  delegate.executeUpdate(sql, columnNames);
        
        long stop = System.currentTimeMillis();
        JDBCStatWrapperMonitor.incrementTimesExecuted(query, params, stop - start, delegate.hashCode());
        
        return ret;
    }

    public int executeUpdate(String sql) throws SQLException {
        if (!sql.equals(query)) {
            query = sql;
            JDBCStatWrapperMonitor.incrementStatementsOpened(query, delegate.hashCode());
        }
        long start = System.currentTimeMillis();
        
        int ret =   delegate.executeUpdate(sql);
        
        long stop = System.currentTimeMillis();
        JDBCStatWrapperMonitor.incrementTimesExecuted(query, params, stop - start, delegate.hashCode());
        
        return ret;
    }

    public Connection getConnection() throws SQLException {
        if(delegate.getConnection() instanceof StatConnection) {
            return delegate.getConnection();
        }
        
        return new StatConnection(delegate.getConnection());
    }

    public int getFetchDirection() throws SQLException {
        return delegate.getFetchDirection();
    }

    public int getFetchSize() throws SQLException {
        return delegate.getFetchSize();
    }

    public ResultSet getGeneratedKeys() throws SQLException {
        return new StatResultSet(delegate.getGeneratedKeys(), query, params);
    }

    public int getMaxFieldSize() throws SQLException {
        return delegate.getMaxFieldSize();
    }

    public int getMaxRows() throws SQLException {
        return delegate.getMaxRows();
    }

    public boolean getMoreResults() throws SQLException {
        return delegate.getMoreResults();
    }

    public boolean getMoreResults(int current) throws SQLException {
        return delegate.getMoreResults(current);
    }

    public int getQueryTimeout() throws SQLException {
        return delegate.getQueryTimeout();
    }

    public ResultSet getResultSet() throws SQLException {
        return new StatResultSet(delegate.getResultSet(), query, params);
    }

    public int getResultSetConcurrency() throws SQLException {
        return delegate.getResultSetConcurrency();
    }

    public int getResultSetHoldability() throws SQLException {
        return delegate.getResultSetHoldability();
    }

    public int getResultSetType() throws SQLException {
        return delegate.getResultSetType();
    }

    public int getUpdateCount() throws SQLException {
        return delegate.getUpdateCount();
    }

    public SQLWarning getWarnings() throws SQLException {
        return delegate.getWarnings();
    }

    public void setCursorName(String name) throws SQLException {
        delegate.setCursorName(name);
    }

    public void setEscapeProcessing(boolean enable) throws SQLException {
        delegate.setEscapeProcessing(enable);
    }

    public void setFetchDirection(int direction) throws SQLException {
        delegate.setFetchDirection(direction);
    }

    public void setFetchSize(int rows) throws SQLException {
        delegate.setFetchSize(rows);
    }

    public void setMaxFieldSize(int max) throws SQLException {
        delegate.setMaxFieldSize(max);
    }

    public void setMaxRows(int max) throws SQLException {
        delegate.setMaxRows(max);
    }

    public void setQueryTimeout(int seconds) throws SQLException {
        delegate.setQueryTimeout(seconds);
    }
}
