/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *  
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *  
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 * 
 */
package com.jimischopp.jyoutube;

import java.io.BufferedInputStream;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.math.BigDecimal;
import java.util.Collection;
import java.util.HashSet;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.apache.commons.httpclient.Cookie;
import org.apache.commons.httpclient.HttpClient;
import org.apache.commons.httpclient.HttpMethod;
import org.apache.commons.httpclient.HttpStatus;
import org.apache.commons.httpclient.cookie.CookiePolicy;
import org.apache.commons.httpclient.methods.GetMethod;
import org.apache.commons.httpclient.methods.PostMethod;
import org.apache.commons.httpclient.methods.multipart.FilePart;
import org.apache.commons.httpclient.methods.multipart.MultipartRequestEntity;
import org.apache.commons.httpclient.methods.multipart.Part;
import org.apache.commons.httpclient.methods.multipart.StringPart;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;


import com.jimischopp.jyoutube.JYouTube.ProgressReportingFilePart.ProgressListener;


/**************************************************************************
 * JYouTube
 * A utility for uploading (and downloading) video content from YouTube
 * programmatically.
 * 
 * This class is a direct Java port of PHPTube by Michael Kamleitner
 * (michael.kamleitner@gmail.com   or   http://www.kamleitner.com/code)
 * 
 * Note that this class depends on:
 * 		commons-httpclient (developed with version 3.1)
 * 		commons-logging (developed with version 1.1)
 * 		commens-codec (dependency of logging; developed with version 1.3)
 * 
 * Quite often these files are quite large, so if you implement the "ProgressListener"
 * interface, and add it to the listen of listeners on the JYouTube instance, you can
 * receive updates about the upload progress. This is extremely useful. Also, in the
 * interim, you can set the JYouTube instance itself as its own listener, since it
 * implements the listener and logs to its logger (ie. myJYTube.addListener(myJYTube))
 * 
 * @version 1.0
 * @author <a href="mailto:jimi@jimischopp.com">James Schopp</a>
 * @date 2007/11/05
 * @www http://www.jimischopp.com/jimischopp.com/Software.html
 * @copyright copyright 2008 - James Schopp
 * 
 */
public class JYouTube implements ProgressListener {
	
	protected final static Log log = LogFactory.getLog(JYouTube.class);
	
	/**
	 * A list of video content categories as recognized by YouTube. These
	 * were discovered empirically by looking at the catid value right there
	 * on the YouTube webpage.
	 */
	public enum Catgeory {
		FILM_AND_ANIMATION(1),
		AUTO_AND_VEHICLE (2),
		MUSIC(10),
		PETS_AND_ANIMALS(15),
		SPORTS(17),
		TRAVEL_AND_PLACES(19),
		GADGETS_AND_GAMES(20),
		PEOPLE_AND_BLOGS(22),
		COMEDY(23),
		ENTERTAINMENT(24),		
		NEWS_AND_POLITICS(25),
		HOWTO_AND_DIY(26);
		
	    private final int catId;
	    Catgeory(int id) {
	        this.catId = id;	       
	    }
		public int getCatId() {
			return catId;
		}
	}

	
	
	
	public final static String YT_COOKIE_LOGININFO 			= "LOGIN_INFO";
	public final static String YT_URL_LOGIN_USERNAME_KEY 	= "$USERNAME$";
	public final static String YT_URL_LOGIN_PASSWORD_KEY 	= "$PASSWORD$";
	public final static String YT_URL_VIDEOID_KEY 			= "$VIDEOID$";
	public final static String YT_URL_VIDEOIDTEMP_KEY 		= "$TEMPID$";	
	public final static String YT_URL_ROOT 					= "http://www.youtube.com:80";
	public final static String YT_URL_LOGIN 				= YT_URL_ROOT + "/login?username="+YT_URL_LOGIN_USERNAME_KEY+"&password="+YT_URL_LOGIN_PASSWORD_KEY+"&next=/index&current_form=loginForm&action_login=1";
	public final static String YT_URL_VIEW 					= YT_URL_ROOT + "/watch?v="+YT_URL_VIDEOID_KEY;
	public final static String YT_URL_GET 					= YT_URL_ROOT + "/get_video?video_id="+YT_URL_VIDEOID_KEY + "&t=" + YT_URL_VIDEOIDTEMP_KEY;
	public final static String YT_URL_PUT_MYVIDEOUPLOAD 	= YT_URL_ROOT + "/my_videos_upload";
	public final static String YT_URL_MYVIDEOS 				= YT_URL_ROOT + "/my_videos";
	
	//we lie about the user-agent, so that YouTube doesn't know we are actually faking them out...
	public final static String USER_AGENT = "Mozilla/4.0 (compatible; MSIE 6.0; Windows NT 5.0)";
	
	protected HttpClient httpclient; 
	protected boolean auth = false;
		
	/**
	 * Get the underlying HttpClient object. This is useful if you need to configure
	 * your own proxy configuration (for example), or specify any unusual connection
	 * properties.
	 * 
	 */
	protected HttpClient getHttpClient()
	{
		return httpclient;
	}
	
	
	/**
	 * Create a new JYouTube object with no user/pass. You can download with this, but you will not be able to upload
	 * new video content.
	 * 
	 * @throws Exception
	 */
	public JYouTube () throws Exception {
		this(null, null);
	}
	
	
	/**
	 * Set a proxy host for the underlying HttpClient object. Besides the "usual" reasons for using a proxy, this
	 * is primarily useful if/when debugging HTTP requests using the HTTPTracer tool in "proxy mode".
	 * 
	 * @param proxyHostName
	 * @param proxyPort
	 */
	public void setProxy(String proxyHostName, int proxyPort) {
		this.httpclient.getHostConfiguration().setProxy(proxyHostName, proxyPort);
	}


	/**
	 * Create a new JYouTube object with the specified username/password. This
	 * object is capable of uploading new video content.
	 * 
	 * @param username
	 * @param password
	 * @throws Exception
	 */
	public JYouTube (String username, String password) throws Exception
	{
		this.httpclient = getNewHttpClient();
		
		
		if (isNullOrEmpty(username) || isNullOrEmpty(password)) {
		 	this.auth = false;		
		 	log.debug("No user/pass specified. Uploading will be disabled.");
		 	return;
		}
		
		//get the login url, with the user/pass substitued
		String url = getYouTubeLoginUrl(username, password);
		HttpMethod method = new GetMethod(url);
		method.setFollowRedirects(true);
		method.setRequestHeader("User-Agent", USER_AGENT);
		try {
			//execute the request!		
			log.debug("Making request to: " + url);
			int result = httpclient.executeMethod(method);
			if (HttpStatus.SC_OK != result) {
				throw new Exception("HTTP Exception + " + result + " " + method.getStatusLine().toString());			
			}
		} finally {
			method.releaseConnection();
		}
		
		//now handle the result!
		this.auth = true;
		Cookie[] cookies = httpclient.getState().getCookies();
		
		boolean success = false;
		for(int i=0; i<cookies.length; i++) {
			Cookie cookie = cookies[i];
			if (YT_COOKIE_LOGININFO.equals(cookie.getName()) && !isNullOrEmpty(cookie.getValue())) {
				success = true;
				break;
			}	
		}
		
		if (!success) {
			auth=false;
			throw new Exception("Login failed!");
		}
	}
	
	
	/**
	 * Download existing video content to the specified local filename.
	 * The video will be in FlashVideo "flv" format.
	 * 
	 * @param videoId
	 * @param videoFilename
	 * @throws Exception
	 */
	public void download (String videoId, String videoFilename) throws Exception
	{	
		String url = getYouTubeViewUrl(videoId);		
		HttpMethod method = new GetMethod(url);
		method.setFollowRedirects(true);
		method.setRequestHeader("User-Agent", USER_AGENT);
		
		
		//first, we get the typical HTML page that YouTube sends back when you ask to view a video...
		String pageHtml = null;
		try {
			log.debug("Making request to: " + url);
			int result = httpclient.executeMethod(method);
			if (result != HttpStatus.SC_OK ) {
				throw new Exception("HTTP Exception + " + result + " " + method.getStatusLine().toString());			
			}
			
			pageHtml = method.getResponseBodyAsString();
			
			if (isNullOrEmpty(pageHtml)) {
				throw new Exception("Empty response from YouTube server!");
			}
		} finally {
			method.releaseConnection();
		}
		
		//then we parse that HTML looking for the URL to actual video file itself...
		//and we download THAT!
		//
		//the javascript in the html page contains a little snippet like:
		//	
		//		t:'OEgsToPDskI4C0fP6X5q-ACZ179S0CT5'
		//
		//our objective it to make a NEW URL like:
		//	http://www.youtube.com/get_video?video_id=S9keLNlvFck&t=OEgsToPDskI4C0fP6X5q-ACZ179S0CT5
		
		// (FYI, I don't speak german so I have no idea what this video is actually ABOUT! I just kinda chose it random...)
		
		Matcher m = pTEMPID.matcher(pageHtml);
		if (!m.find() || m.groupCount() != 2) {
			throw new Exception("Could not find video download URL in view page.");
		}
		String tempId = m.group(1);
		url = getYouTubeGetUrl(videoId, tempId);
		
		//ok, now we need a file to write to
		File f = new File(videoFilename);		
		FileOutputStream fos = new FileOutputStream(f);
		
		method = new GetMethod(url);
		try {
			log.debug("Making request to: " + url);
			int result = httpclient.executeMethod(method);
			if (result != HttpStatus.SC_OK ) {
				throw new Exception("HTTP Exception + " + result + " " + method.getStatusLine().toString());		
			}
			
			InputStream is = new BufferedInputStream(method.getResponseBodyAsStream());			 
			for (byte[] buff = new byte[1024 * 10]; -1!=is.read(buff); fos.write(buff));
			is.close();
		} finally {
			method.releaseConnection();
			fos.close();
		}
	}
	private final static Pattern pTEMPID = Pattern.compile("t:'(([a-zA-Z0-9_\\-]+))'");
	
	
	
	
	/**
	 * 
	 * Upload new video content. Note that after a successful upload, your video will still not be immediately available
	 * since the YouTube servers must transcode your format to "flv" Flash Video format first. Only then will your
	 * new videoId be valid.
	 * 
	 * @param videoFilename name of the local file to be uploaded
	 * @param videoTitle Video-Title on YouTube
	 * @param videoTags blank-separated list of keywords to tag the video content with for later search
	 * @param videoDescription description of the video
	 * @param cat category of the video content (sports, comedy, etc...)
	 * @param videoLanguage language of the video (DE, EN, etc...)
	 * @param isPublic if this video is viewable by the general public or not
	 * @return the videoId of the new content on YouTube
	 * @throws Exception
	 */
	public String upload (String videoFilename, String videoTitle, String videoTags, String videoDescription, JYouTube.Catgeory cat, String videoLanguage, boolean isPublic) throws Exception
	{
		if (!auth) {
			throw new Exception("Not authenticated!");
		}
		
		File videoFile = new File(videoFilename);
		if (!videoFile.exists() || !videoFile.isFile() || !videoFile.canRead()) {
			throw new IOException("File '"+videoFilename+"' must be an existing reabable file in order to upload to YouTube!");
		}
		
		
		
		//GET THE START UPLOAD PAGE AND FILL IT OUT!!				
		PostMethod method = new PostMethod(YT_URL_PUT_MYVIDEOUPLOAD);
		method.addRequestHeader("User-Agent",USER_AGENT);
		method.addParameter("field_myvideo_title",videoTitle);
		method.addParameter("field_myvideo_keywords", videoTags);
		method.addParameter("field_myvideo_descr",videoDescription);
		method.addParameter("language",videoFilename);
		method.addParameter("field_myvideo_categories",cat.getCatId()+"");
		method.addParameter("ignore_broadcast_settings","0");
		method.addParameter("action_upload","1");
		method.addParameter("field_privacy",isPublic?"public":"private");
			
		String pageHtml;
		try {
			int result = httpclient.executeMethod(method);
			switch (result) {
				case HttpStatus.SC_OK:
					break;
					
				default:
					throw new Exception("HTTP Exception + " + result + " " + method.getStatusLine().toString());		
			}
			
			pageHtml = method.getResponseBodyAsString();
			
			//FileOutputStream fos = new FileOutputStream("YouTubeView.html");
			//fos.write(pageHtml.getBytes());
			//fos.close();
			
			if (isNullOrEmpty(pageHtml)) {
				throw new Exception("Empty response from YouTube server!");
			}
			
		} finally {
			method.releaseConnection();
		}
		
		
		 
		
		
			
		
		//ok, so we have a ton of HTML. We are looking for a form like:
		//
		//	<form method="post" enctype="multipart/form-data" action="http://lax-v103.lax.youtube.com/my_videos_post" name="theForm" id="theForm" onSubmit="return formValidator();">
		//       ...lots of hidden inputs...
		//       <input type="hidden" name="addresser" value="pxSwr2P2pxJwM6jkcgKSjtyRJ57A2LJmbMPs-FhQp3LwaDPmk9SIuDPz_CiFoaAH-ufrSjmM9g9hOmjyZ6bLiQ==">
		//       ...nore hidden inputs...
		//
		// we need to get the "action" URL, and the addresser value, and submit the form with our video file...
		// That is the obejctive of all the stupid parsing code below:
		
		int byteOffset = pageHtml.indexOf("id=\"theForm\"");
		if (byteOffset == -1) throw new Exception("Unexpected repsonse from YouTube server! We did not receive the upload page we expected.");
		
		byteOffset = pageHtml.lastIndexOf("<form", byteOffset);
		byteOffset = pageHtml.indexOf("action=\"", byteOffset) + 8;
		String url = pageHtml.substring(byteOffset, pageHtml.indexOf("\"", byteOffset));
		
		byteOffset = pageHtml.indexOf("name=\"addresser\" value=\"", byteOffset) + 24;
		String addresser = pageHtml.substring(byteOffset, pageHtml.indexOf("\"", byteOffset));
		
		
		
		
		
		//UPLOAD THE ACTUAL VIDEO FILE!!
		method = new PostMethod(url);
		method.addRequestHeader("User-Agent",USER_AGENT);		
        method.setRequestEntity(new MultipartRequestEntity(new Part[]{
        							new StringPart("field_command", "myvideo_submit"),
        							new StringPart("field_myvideo_title",videoTitle),
        							new StringPart("field_myvideo_keywords",videoTags),
        							new StringPart("field_myvideo_descr",videoDescription),
        							new StringPart("language",videoLanguage),
        							new StringPart("field_myvideo_categories",cat.getCatId()+""),
        							new StringPart("action_upload","1"),
        							new StringPart("addresser",addresser),					
        							new StringPart("field_privacy",isPublic?"public":"private"),        							
				        			new ProgressReportingFilePart("field_uploadfile", videoFile, this.listeners),
				        		}, 
				        		method.getParams()));
				
		try {
			int result = httpclient.executeMethod(method);
			switch (result) {
				case HttpStatus.SC_NOT_FOUND:
				case HttpStatus.SC_INTERNAL_SERVER_ERROR:
					break;
					
				default:
				//	throw new Exception("HTTP Exception + " + result + " " + method.getStatusLine().toString());		
			}
			
		} finally {
			method.releaseConnection();
		}
		
		
		//ok, now we need to go back to our homepage, and get the video ID of the new video we just uploaded...
		method = new PostMethod(YT_URL_MYVIDEOS);
		try {
			int result = httpclient.executeMethod(method);
			switch (result) {
				case HttpStatus.SC_OK:
					break;
					
				default:
					throw new Exception("HTTP Exception + " + result + " " + method.getStatusLine().toString());		
			}
			
			pageHtml = method.getResponseBodyAsString();
			
			//FileOutputStream fos = new FileOutputStream("YouTubeView.html");
			//fos.write(pageHtml.getBytes());
			//fos.close();
			
			if (isNullOrEmpty(pageHtml)) {
				throw new Exception("Empty response from YouTube server!");
			}
			
		} finally {
			method.releaseConnection();
		}
		
		byteOffset = pageHtml.indexOf("id=\"checkbox_")+13;
		String videoId = pageHtml.substring(byteOffset, pageHtml.indexOf("\"", byteOffset));
		return videoId;
	}

	private static boolean isNullOrEmpty(String str)
	{
		if (str==null) return true;
		return str.length() == 0;
	}
	
	protected static String getYouTubeLoginUrl(String username, String password)
	{
		return YT_URL_LOGIN.replace(YT_URL_LOGIN_USERNAME_KEY, username).replace(YT_URL_LOGIN_PASSWORD_KEY, password);
	}
	protected static String getYouTubeViewUrl(String videoId)
	{
		return YT_URL_VIEW.replace(YT_URL_VIDEOID_KEY, videoId);
	}
	protected static String getYouTubeGetUrl(String videoId, String tempId)
	{
		return YT_URL_GET.replace(YT_URL_VIDEOID_KEY, videoId).replace(YT_URL_VIDEOIDTEMP_KEY, tempId);
	}
	
	
	protected static HttpClient getNewHttpClient()
	{
		HttpClient tmpHttpclient = new HttpClient();
		tmpHttpclient.getHttpConnectionManager().getParams().setConnectionTimeout(1 * 1000 * 60 * 60); //1hr. Videos can be huge!
		tmpHttpclient.getParams().setCookiePolicy(CookiePolicy.DEFAULT);
		
		return tmpHttpclient;
	}	
	
	/**
	 * 
	 * ProgressReportingFilePart issues events to listeners updating them about the progress of a multi-part file upload
	 * of an HTTP request. This is useful for monitoring the progress is very large file uploads, such as YouTube videos...
	 *
	 */
	public static class ProgressReportingFilePart extends FilePart
	{	
		
		/**
		 * 
		 * Interface to be implemented if you wish to receive updates about HTTP file upload progress
		 *
		 */
		public static interface ProgressListener
		{
			public void progressUpdate(ProgressUpdateEvent event);
		}
		
		/**
		 *  Contains event information about the progress of an HTTP File Upload
		 *
		 */
		public static class ProgressUpdateEvent
		{
			protected long totalBytesToSend;
			protected long totalBytesSent;
			protected String errorMessage;
			protected boolean error;
			protected String fileName;
			protected String parameterName;
			protected long elapsedMillis;
			
			public boolean isError() {
				return error;
			}
			public String getErrorMessage() {
				return errorMessage;
			}
			public long getTotalBytesSent() {
				return totalBytesSent;
			}
			public long getTotalBytesToSend() {
				return totalBytesToSend;
			}
			public long getElapsedMillis() {
				return elapsedMillis;
			}
			public String getFileName() {
				return fileName;
			}
			public String getParameterName() {
				return parameterName;
			}
			protected ProgressUpdateEvent(long totalBytesToSend, long totalBytesSent, long elapsedMillis, String errorMessage, boolean error, String fileName, String parameterName) {
				super();
				this.totalBytesToSend = totalBytesToSend;
				this.totalBytesSent = totalBytesSent;
				this.elapsedMillis = elapsedMillis;
				this.errorMessage = errorMessage;
				this.error = error;
				this.fileName = fileName;
				this.parameterName = parameterName;
			}
		}

		protected String parameterName;
		protected String fileName;
		protected Collection<ProgressListener> listeners =  new HashSet<ProgressListener>();
		
		@Override
		protected void sendData(OutputStream out) throws IOException {			
			log.trace("enter sendData(OutputStream out)");
	        if (lengthOfData() == 0) {
	            
	            // this file contains no data, so there is nothing to send.
	            // we don't want to create a zero length buffer as this will
	            // cause an infinite loop when reading.
	            log.debug("No data to send.");
	            return;
	        }
	        
	        
	        //some ars for tracking our upload progress...
	        long bytesSent = 0;
	        long totalLength = this.getSource().getLength();
	        long logEveryXBytes = totalLength / 100; //notify every 1%...
	        long logEveryXBytesNotified = 0;
	        long timeStart = System.currentTimeMillis();
	        
	        byte[] tmp = new byte[4096];
	        InputStream instream = this.getSource().createInputStream();
	        try {
	            int len;
	            while ((len = instream.read(tmp)) >= 0) {
	                out.write(tmp, 0, len);
	                
	                //track the upload and see if we need to notify people...
	                bytesSent += len;
	                if (bytesSent >= (logEveryXBytesNotified+logEveryXBytes)) {	             
	                	logEveryXBytesNotified = bytesSent;
	                	ProgressUpdateEvent event = new ProgressUpdateEvent(totalLength, bytesSent, System.currentTimeMillis()-timeStart, null, false, fileName, parameterName);
	                	notifyListeners(event);
	                }
	            }
	            
	            //ok, we're finished. Let them know...
	            ProgressUpdateEvent event = new ProgressUpdateEvent(totalLength, bytesSent, System.currentTimeMillis()-timeStart, null, false, fileName, parameterName);
            	notifyListeners(event);
            	
	        } catch (IOException e) {
	        	//we got an error, let them know...
	        	ProgressUpdateEvent event = new ProgressUpdateEvent(totalLength, bytesSent, System.currentTimeMillis()-timeStart, e.getMessage(), true, fileName, parameterName);
            	notifyListeners(event);
            	throw e;
            	
	        } finally {
	            // we're done with the stream, close it
	            instream.close();
	        }
		}
		
		public void addListener(ProgressListener listener)
		{
			if (listener==null) return;
			synchronized (listeners) {
				listeners.add(listener);
			}
		}
		
		public void addListener(Collection<ProgressListener> listener)
		{
			if (listener==null) return;
			synchronized (listeners) {
				listeners.addAll(listener);
			}
		}
		
		public boolean removeListener(ProgressListener listener)
		{
			if (listener==null) return true;
			synchronized (listeners) {
				return listeners.remove(listener);
			}
		}
		
		protected void notifyListeners(ProgressUpdateEvent event) {
			synchronized (this.listeners) {
				for(ProgressListener listener : this.listeners) {
					try {
						listener.progressUpdate(event);
					} catch (Exception e) {}
				}
			}
		}

		public ProgressReportingFilePart(String parameterName, File file, Collection<ProgressListener> progressListeners) throws FileNotFoundException {
			super(parameterName, file);
			this.parameterName = parameterName;
			this.fileName = file.getName();
			this.addListener(progressListeners);
		}

		public String getFileName() {
			return fileName;
		}

		public String getParameterName() {
			return parameterName;
		}
	}
	
	
	/**
	 * Collection of ProgressListeners. Whenever a file is uploaded, these listeners will
	 * all receive updates
	 */
	protected Collection<ProgressListener> listeners =  new HashSet<ProgressListener>();
	
	/**
	 * Add a new ProgressListener to the list of listeners to receive HTTP upload progress updates
	 * @param listener
	 */
	public void addListener(ProgressListener listener)
	{
		if (listener==null) return;
		synchronized (listeners) {
			listeners.add(listener);
		}
	}
	
	/**
	 * Remove a listener from the list of HTTP upload progress update listeners
	 * @param listener
	 * @return
	 */
	public boolean removeListener(ProgressListener listener)
	{
		if (listener==null) return true;
		synchronized (listeners) {
			return listeners.remove(listener);
		}
	}

	
	/**
	 * Default implementation of the ProgressListener interface, useful for logging when the calling app has not
	 * yet implemented the listener interface.
	 */
	public void progressUpdate(com.jimischopp.jyoutube.JYouTube.ProgressReportingFilePart.ProgressUpdateEvent event) {
		if(event.isError()) {
			if(log.isInfoEnabled()) {
				log.info("Element '"+event.getParameterName()+"' (file '"+event.getFileName()+"'): error '"+event.getErrorMessage()+"'");
			}
		} else {
			if(log.isInfoEnabled()) {
				long bytesPerMilli = event.getTotalBytesSent() / event.getElapsedMillis();
				long millisRemaining = (event.getTotalBytesToSend() - event.getTotalBytesSent()) / bytesPerMilli;
				log.info("Element '"+event.getParameterName()+"' " +
								   "(file '"+ event.getFileName()+"'): " +
								   	"uploaded '"+event.getTotalBytesSent()+"' of '"+event.getTotalBytesToSend()+"' bytes: " +
								   		new BigDecimal(((double)bytesPerMilli*1000D)/1024D).setScale(2, BigDecimal.ROUND_HALF_UP).toString() + "KB/s " +
								   		"(" + (millisRemaining/1000) + "s remaining)"
								   );
			}
		}
	}
		
}
