/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *  
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *  
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 * 
 *
 *
 *
 *
 * A JNI wrapper around the MS DPAPI Encryption library. Encrypts/Decrypts strings using the
 * Microsoft DPAPI library (in Crypt32.dll). This has the restriction that the original string
 * must be decrypted on the same machine it was encrypted on, since the MSDPAPIJava library
 * uses the Machine-level store.
 * <p>
 * Copyright 2004, James Schopp
 *
 * @author James Schopp
 *
 */
// MSDPAPIJava.cpp : Defines the entry point for the DLL application.
//

//#define WIN32_LEAN_AND_MEAN		// Exclude rarely-used stuff from Windows headers
#include <windows.h>
#include <Wincrypt.h>

#include "MSDPAPIJava.h"



BOOL APIENTRY DllMain( HANDLE hModule, 
                       DWORD  ul_reason_for_call, 
                       LPVOID lpReserved
					 )
{
    switch (ul_reason_for_call)
	{
		case DLL_PROCESS_ATTACH:
		case DLL_THREAD_ATTACH:
		case DLL_THREAD_DETACH:
		case DLL_PROCESS_DETACH:
			break;
    }
    return TRUE;
}


static const char          fillchar = '=';
static const unsigned char np = 0;

                                   // 0000000000111111111122222222223333333333444444444455555555556666
                                   // 0123456789012345678901234567890123456789012345678901234567890123
static const char Base64Table[] = {'A','B','C','D','E','F','G','H','I','J','K','L','M','N','O','P','Q','R','S','T','U','V','W','X','Y','Z',
                                   'a','b','c','d','e','f','g','h','i','j','k','l','m','n','o','p','q','r','s','t','u','v','w','x','y','z',
                                   '0','1','2','3','4','5','6','7','8','9','+','/'};

// Decode Table gives the index of any valid base64 character in the Base64 table]
// 65 == A, 97 == a, 48 == 0, 43 == +, 47 == /

                                     // 0  1  2  3  4  5  6  7  8  9 
static const unsigned char DecodeTable[] = {np,np,np,np,np,np,np,np,np,np,  // 0 - 9
                                       np,np,np,np,np,np,np,np,np,np,  //10 -19
                                       np,np,np,np,np,np,np,np,np,np,  //20 -29
                                       np,np,np,np,np,np,np,np,np,np,  //30 -39
                                       np,np,np,62,np,np,np,63,52,53,  //40 -49
                                       54,55,56,57,58,59,60,61,np,np,  //50 -59
                                       np,np,np,np,np, 0, 1, 2, 3, 4,  //60 -69
                                       5, 6, 7, 8, 9,10,11,12,13,14,   //70 -79
                                       15,16,17,18,19,20,21,22,23,24,  //80 -89
                                       25,np,np,np,np,np,np,26,27,28,  //90 -99
                                       29,30,31,32,33,34,35,36,37,38,  //100 -109
                                       39,40,41,42,43,44,45,46,47,48,  //110 -119
                                       49,50,51,np,np,np,np,np,np,np,  //120 -129
                                       np,np,np,np,np,np,np,np,np,np,  //130 -139
                                       np,np,np,np,np,np,np,np,np,np,  //140 -149
                                       np,np,np,np,np,np,np,np,np,np,  //150 -159
                                       np,np,np,np,np,np,np,np,np,np,  //160 -169
                                       np,np,np,np,np,np,np,np,np,np,  //170 -179
                                       np,np,np,np,np,np,np,np,np,np,  //180 -189
                                       np,np,np,np,np,np,np,np,np,np,  //190 -199
                                       np,np,np,np,np,np,np,np,np,np,  //200 -209
                                       np,np,np,np,np,np,np,np,np,np,  //210 -219
                                       np,np,np,np,np,np,np,np,np,np,  //220 -229
                                       np,np,np,np,np,np,np,np,np,np,  //230 -239
                                       np,np,np,np,np,np,np,np,np,np,  //240 -249
                                       np,np,np,np,np,np               //250 -256
									};


char * base64Encode(const unsigned char* data, const unsigned long len)
{
    unsigned long      i;
    char               c;    
    char *             ret;

    unsigned long      indexIntoRet = 0;
    ret = new char[len * 2];

    for (i = 0; i < len; ++i)
    {
        c = (data[i] >> 2) & 0x3f;
        ret[indexIntoRet] = Base64Table[c];
        indexIntoRet++;

        c = (data[i] << 4) & 0x3f;
        if (++i < len)
            c |= (data[i] >> 4) & 0x0f;

        ret[indexIntoRet] = Base64Table[c];
        indexIntoRet++;

        if (i < len)
        {
            c = (data[i] << 2) & 0x3f;
            if (++i < len)
                c |= (data[i] >> 6) & 0x03;

            ret[indexIntoRet] = Base64Table[c];
            indexIntoRet++;
        }
        else
        {
            ++i;
            ret[indexIntoRet] = fillchar;
            indexIntoRet++;            
        }

        if (i < len)
        {
            c = data[i] & 0x3f;
            ret[indexIntoRet] = Base64Table[c];
            indexIntoRet++;
        }
        else
        {
            ret[indexIntoRet] = fillchar;
            indexIntoRet++;
        }
    }

    ret[indexIntoRet] = 0;
    return ret;
}

unsigned char * base64Decode(const char* data, const unsigned long len, unsigned long *outLen)
{
    unsigned long      i;
    char               c;
    char               c1;
    
    unsigned char*     ret;
    
    long      indexIntoRet = 0;
    ret = new unsigned char[len * 2];

    for (i = 0; i < len; ++i)
    {
        c = (char) DecodeTable[(unsigned char)data[i]];
        ++i;
        c1 = (char) DecodeTable[(unsigned char)data[i]];
        c = (c << 2) | ((c1 >> 4) & 0x3);
        
        ret[indexIntoRet] = c;
        indexIntoRet++;

        if (++i < len)
        {
            c = data[i];
            if (fillchar == c)
                break;

            c = (char) DecodeTable[(unsigned char)data[i]];
            c1 = ((c1 << 4) & 0xf0) | ((c >> 2) & 0xf);
            ret[indexIntoRet] = c1;
            indexIntoRet++;;
        }

        if (++i < len)
        {
            c1 = data[i];
            if (fillchar == c1)
                break;

            c1 = (char) DecodeTable[(unsigned char)data[i]];
            c = ((c << 6) & 0xc0) | c1;
            ret[indexIntoRet] = c;
            indexIntoRet++;
        }
    }
    
    ret[indexIntoRet] = 0;
    outLen[0] = indexIntoRet;
    return(ret);
}





/*
 * Class:     com_jimischopp_msdpapi_MSDPAPIJava
 * Method:    CryptProtectData
 * Signature: (Ljava/lang/String;)Ljava/lang/String;
 */
JNIEXPORT jstring JNICALL Java_com_jimischopp_msdpapi_MSDPAPIJava_CryptProtectData
  (JNIEnv *env, jclass obj, jstring dataIn)
{
    //return value!
    jstring jstrRet;

    //setup the input string...
    DATA_BLOB DataIn;
	DataIn.pbData = (BYTE *)(env->GetStringUTFChars(dataIn, 0));    

    //if we run out of mem, then GetStringUTFChars throws an OutOfMem exception. We immediately
    //return null here to let the OOM exception propograte.
    if (DataIn.pbData == NULL) {
        return NULL;
    }
    DataIn.cbData = (DWORD)strlen((const char*)DataIn.pbData) + 1;

    //we'll receive stuff in the out...
    DATA_BLOB DataOut;
    
    //encrypt!!
    if(CryptProtectData(
		 &DataIn,
		 (LPCWSTR)"W2KDummy",
		 NULL,                               // Optional entropy not used.
		 NULL,                               // Reserved.
		 NULL,                               // Pass NULL for the prompt structure.
		 CRYPTPROTECT_UI_FORBIDDEN | CRYPTPROTECT_LOCAL_MACHINE,
		 &DataOut))
	{
		 //worked! So now we need to Base64 encode it...
        char* strEncrypted = base64Encode((const unsigned char *)(DataOut.pbData), DataOut.cbData);
                
        jstrRet = env->NewStringUTF(strEncrypted);
        delete strEncrypted;
        LocalFree(DataOut.pbData);

	} else {
		//oh-oh!
        jstrRet = NULL;
	}

    //release the UTFs
    env->ReleaseStringUTFChars(dataIn, (const char*)DataIn.pbData);

    return jstrRet;
}



/*
 * Class:     com_jimischopp_msdpapi_MSDPAPIJava
 * Method:    CryptUnprotectData
 * Signature: (Ljava/lang/String;)Ljava/lang/String;
 */
JNIEXPORT jstring JNICALL Java_com_jimischopp_msdpapi_MSDPAPIJava_CryptUnprotectData
  (JNIEnv *env, jclass obj, jstring dataIn)
{
    jstring   jstrRet;
    DATA_BLOB DataIn;
    DATA_BLOB DataOut;
    LPWSTR   strDesc = NULL;

    //first we need to get a C-string from the J-string
    const char * strEncoded = env->GetStringUTFChars(dataIn, 0);
    if (strEncoded == NULL) {
        return NULL; //just return to throw the OutOfmemory exception...
    }

    //then base64Decode it...
    DataIn.pbData = base64Decode(strEncoded, (unsigned long) strlen(strEncoded), &DataIn.cbData);
    
    //now decrypt it!
    if (CryptUnprotectData(
                &DataIn,
                &strDesc,
                NULL,
                NULL,
                NULL,
                CRYPTPROTECT_UI_FORBIDDEN | CRYPTPROTECT_LOCAL_MACHINE,
                &DataOut))
    {
        //we're good!
        LocalFree(strDesc);
        //copy it to a new location and put a NULL on the end...
        char * strDecrypted = new char[DataOut.cbData+1];
        memcpy(strDecrypted, DataOut.pbData, DataOut.cbData);
        strDecrypted[DataOut.cbData] = NULL;

        //get a j-string from it...
        jstrRet = env->NewStringUTF(strDecrypted);

        //release the returned copy and our temp copy...
        LocalFree(DataOut.pbData);
        delete strDecrypted;

    } else {
        return NULL;
    }

    //release the copy of decoded-ecrypted...
    delete DataIn.pbData;

    //finally, release the copy of UTFs
    env->ReleaseStringUTFChars(dataIn, strEncoded);

	return jstrRet;
}

