/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *  
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *  
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 * 
 */
package com.jimischopp.checkstyle;

import com.puppycrawl.tools.checkstyle.api.Check;
import com.puppycrawl.tools.checkstyle.api.DetailAST;
import com.puppycrawl.tools.checkstyle.api.TokenTypes;



/**
 * An abstract class the proves useful methods for decoding/deciphering AST blocks.
 *  
 * <p>
 * Copyright 2003, James Schopp
 *
 * @author James Schopp
 * @see com.puppycrawl.tools.checkstyle.api.Check
 * @since Oct 27, 2003
 * 
 */
public abstract class AbstractBaseChecker extends Check {

	/**
     * Find the name of the variable that a value was a assigned to. Given an expression like
     *          <pre> int x = 5; </pre>
     * where the AST represents a VARIABLE_DEF or ASSIGN block, return the string "x"
	 * @param astNewBlock
	 * @return the name of the variable receiving the value
	 */
    protected String getRefNameAssigned(DetailAST astNewBlock)
    {
        DetailAST astRefOp = getParentType(astNewBlock, TokenTypes.VARIABLE_DEF);        
        if (astRefOp != null) {
            return astRefOp.findFirstToken(TokenTypes.IDENT).getText().trim();
        }
         
        astRefOp = getParentType(astNewBlock, TokenTypes.ASSIGN);
        if (astRefOp != null) {
            return astRefOp.findFirstToken(TokenTypes.IDENT).getText().trim();
        }
        
        return "";            
    }
    
    
    /**
     * Find the nearest of parent of the specified type.
     * Image you have the following expression
     *          <pre> int x = 5; </pre>
     * and you have a the AST "ASSING" block. You can find the encompassing VARIABLE_DEF by calling
     *          <pre> getParentType(astAssingBlock, ToeknTypes.VARIABLE_DEF); </pre>
     * or even the encompassing method by:
     *          <pre> getParentType(astAssingBlock, ToeknTypes.METHOD_DEF); </pre>
     * 
     * @param ast
     * @param type
     * @return the nearest of parent of the specified type.
     */
    protected DetailAST getParentType(DetailAST ast, final int type)
    {
        while (ast!=null  &&  ast.getType()!=type) {
            ast = ast.getParent();
        }
        
        return ast;            
    }
    
    
    /**
     * Get a string representation of the specified ast block. This method decodes the different
     * AST types differently depending on the type. Most importantly:
     * <p>
     * given the expression
     *      <pre> new java.util.Date().getTime() </pre>
     * 
     * the LITERAL_NEW block will return something like: java.sql.Date<p>
     * the DOT block will return something like java.sql.Date<p>
     * the METHOD_CALL block will return something like java.sql.Date.getTime<p>
     * etc...<p>
     * 
     * You will often need to parse for String.lastIndexOf('.') to find the "real"
     * string you are actually interested in.
     * 
     * @param ast
     * @return a string representation of the specified ast block
     */
    protected String toString(DetailAST ast)
    {
        String strRet;
        switch(ast.getType()) {
            case TokenTypes.LITERAL_NEW:
                strRet = toString_LITERAL_NEW(ast);
                break;
                
            case TokenTypes.TYPE:
                strRet = toString_TYPE(ast);
                break;
                
            case TokenTypes.TYPECAST:
                strRet = toString_TYPECAST(ast);
                break;
                
            case TokenTypes.IDENT:
                strRet = toString_IDENT(ast);
                break;
            
            case TokenTypes.METHOD_CALL:
                strRet = toString_METHOD_CALL(ast);
                break;
            
            case TokenTypes.DOT:
                strRet = toString_DOT(ast);
                break;
            
            case TokenTypes.ELIST:
                strRet = toString_ELIST(ast);
                break;
                                
            case TokenTypes.LITERAL_VOID:
                strRet = "void";
                break;
                
            case TokenTypes.LITERAL_BOOLEAN:
                strRet = "boolean";
                break;
                
            case TokenTypes.LITERAL_BYTE:
                strRet = "byte";
                break;
                
            case TokenTypes.LITERAL_CHAR:
                strRet = "byte";
                break;
            
            case TokenTypes.LITERAL_SHORT:
                strRet = "short";
                break;
                
            case TokenTypes.LITERAL_INT:
                strRet = "int";
                break;
                
            case TokenTypes.LITERAL_FLOAT:
                strRet = "float";
                break;
                
            case TokenTypes.LITERAL_LONG:
                strRet = "long";
                break;
                
            case TokenTypes.LITERAL_DOUBLE:
                strRet = "double";
                break;
                
            default:
                strRet = "";
            
        }
        return strRet;        
    }
    
    public boolean isInsideLoop(final DetailAST ast)
	{
		 //FOR 
		DetailAST astLoop = getParentType(ast, TokenTypes.LITERAL_FOR);
        if (astLoop != null) {
            return true;
        }
        
        //DO
        astLoop = getParentType(ast, TokenTypes.LITERAL_DO);
        if (astLoop != null) {
            return true;
        }
        
        
        //WHILE
        astLoop = getParentType(ast, TokenTypes.LITERAL_WHILE);
        if (astLoop != null) {
            return true;
        }
		
        return false;
	}
    
    
    private String toString_DOT(DetailAST ast)
    {
        DetailAST astChild1 = (DetailAST)ast.getFirstChild();
        DetailAST astChild2 = (DetailAST)astChild1.getNextSibling();
                
        return toString(astChild1) + '.' + toString(astChild2);        
    }
    
    private String toString_IDENT(DetailAST ast)
    {
        return ast.getText();        
    }
    
    private String toString_METHOD_CALL(DetailAST ast)
    {
        return toString((DetailAST)ast.getFirstChild());// + '(' + toString_ELIST(ast.findFirstToken(TokenTypes.ELIST)) + ')';        
    }

    private String toString_ELIST(DetailAST ast)
    {
        return " ";
    }
    
    private String toString_LITERAL_NEW(DetailAST ast)
    {
        return toString((DetailAST)ast.getFirstChild());
    }
    
    private String toString_TYPECAST(DetailAST ast)
    {
        return /* '(' +  */ toString((DetailAST)ast.getFirstChild()) /* + ')' */;
    }
    
    private String toString_TYPE(DetailAST ast)
    {
        return toString((DetailAST)ast.getFirstChild());
    }
    
}
