/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *  
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *  
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 * 
 */
package com.jimischopp.checkstyle;


import java.util.regex.Pattern;

import com.puppycrawl.tools.checkstyle.api.DetailAST;
import com.puppycrawl.tools.checkstyle.api.TokenTypes;
import com.puppycrawl.tools.checkstyle.api.Utils;


/**
 * Finds calls to resource objects which should not be performed in loops.
 * For example, assuming your resource is a database object, you should avoid calling queries
 * in loops (better to find some mechanism to call it once on the outside of the loop).
 * <p>
 * The way of "detecting" a resource object is by simply looking at a regular expression on all
 * method calls (is resourceIsParam is false) or on all parameters to method calls (if
 * resourceIsParam is true). If the expression is matched, we then look to see if it is found within a loop
 * structure.
 * 
 * 
 *  <pre>
 *  <module name="com.jimischopp.checkstyle.ResourceUseInLoopCheck">
 *      <property name="format" value="conn.*"/>
 *      <property name="resourceIsParam" value="true"/>
 *  </module> 
 *  </pre>
 *  For example, the above will look for anywhere that a variable named "conn*" is used as parameter to a method call
 *  (as this most likely indicates you are using a DB connection in a loop).
 *  
 *  
 *  
 *  <pre>
 *  <module name="com.jimischopp.checkstyle.ResourceUseInLoopCheck">
 *      <property name="format" value="execute.*"/>
 *      <property name="resourceIsParam" value="false"/>
 *  </module> 
 *  </pre>
 *  The above will look for anywhere that a method name "execute*" is called (regardless of the object it is called on)
 *  (as this most likely indicates you are using a statement in a loop).
 *  
 * 
 * <p>
 * Copyright 2003, James Schopp
 *
 * @author James Schopp
 * @see com.puppycrawl.tools.checkstyle.api.Check
 * @since Oct 27, 2003
 */
public class ResourceUseInLoopCheck extends AbstractBaseChecker {
	
	//public final String DEFAULT_RESOURCE_FORMAT = "execute.*";
	public final String DEFAULT_RESOURCE_FORMAT = "conn.*";
	
	
	/** the regexp to match against */
	private Pattern mRegexp;
	/** the format string of the regexp */
	private String mFormat;
	
	private boolean resourceIsParam = true;
	
	public ResourceUseInLoopCheck()
	{
		setFormat(DEFAULT_RESOURCE_FORMAT);
	}
	
	
	/**
	 * Set the format to the specified regular expression.
	 * @param aFormat a <code>String</code> value
	 */
	public void setFormat(String aFormat)
	{
		mRegexp = Utils.getPattern(aFormat);
		mFormat = aFormat;
	}
	
	/** @return the regexp to match against */
	public Pattern getRegexp()
	{
		return mRegexp;
	}
	
	/** @return the regexp format */
	public String getFormat()
	{
		return mFormat;
	}
	
	
	
	public int[] getDefaultTokens()
	{
		if (resourceIsParam)
			return new int[]{TokenTypes.IDENT, };
		
		return new int[]{TokenTypes.METHOD_CALL, };
	}
	
	public int[] getAcceptableTokens()
	{
		return getDefaultTokens();
	}
	
	public int[] getRequiredTokens()
	{
		return getDefaultTokens();
	}
	
	public void visitToken(final DetailAST ast)
	{
		switch(ast.getType()) {
			case TokenTypes.METHOD_CALL:
				visitToken_METHOD(ast);
				return;
			
			case TokenTypes.IDENT:
				visitToken_PARAMETERS(ast);
				return;
			
		}	
	}
	


	private void visitToken_PARAMETERS(DetailAST ast) {
		
		//see if we have a parent that is an EXPR, and of that has a parent that is a method call
		DetailAST astExpr = getParentType(ast, TokenTypes.EXPR);
		if (astExpr==null) return;
		DetailAST astMeth = getParentType(astExpr, TokenTypes.METHOD_CALL);
        if (astMeth == null) {
            return;
        }
		
		//get the statement that is being called...        
		String strParams = toString(ast);
        
		int indx = strParams.lastIndexOf('.');
		if (indx != -1 ) strParams = strParams.substring(indx+1);
		
		if (!getRegexp().matcher(strParams).matches()) {
			return;
		}
		    
        //now see if we are inside a loop structure...
		if (isInsideLoop(ast)) {
            log(ast, "resourceUseInLoopCheck");
            return;
        }
        
        return;	
	}


	private void visitToken_METHOD(DetailAST ast)
	{
		//get the statement that is being called...        
		String strMethodCall = toString(ast);
        
		int indx = strMethodCall.lastIndexOf('.');
		if (indx != -1 ) strMethodCall = strMethodCall.substring(indx+1);
		
		if (!getRegexp().matcher(strMethodCall).matches()) {
			return;
		}
		    
        //now see if we are inside a loop structure...
		if (isInsideLoop(ast)) {
            log(ast, "resourceUseInLoopCheck");
            return;
        }
        
        return;	
	}
	

	public boolean isResourceIsParam() {
		return resourceIsParam;
	}


	public void setResourceIsParam(boolean resourceIsParam) {
		this.resourceIsParam = resourceIsParam;
	}
	
}
