/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *  
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *  
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 * 
 */
package com.jimischopp.checkstyle;


import com.puppycrawl.tools.checkstyle.api.DetailAST;
import com.puppycrawl.tools.checkstyle.api.TokenTypes;


/**
 * Checks that 'new java.util.Date().getTime()' is never called.
 * This should be replaced by a simple call to System.currentTimeMillis().
 * 
 * <p>
 * Copyright 2003, James Schopp
 *
 * @author James Schopp
 * @see com.puppycrawl.tools.checkstyle.api.Check
 * @since Oct 27, 2003
 *
 */
public class WastefulDateCreationCheck extends AbstractBaseChecker {
	
	public int[] getDefaultTokens()
	{     
		return new int[]{TokenTypes.LITERAL_NEW, };
	}
	
	public int[] getAcceptableTokens()
	{
		return getDefaultTokens();
	}
	
	public int[] getRequiredTokens()
	{
		return getDefaultTokens();
	}
	
	public void visitToken(final DetailAST ast)
	{
				
		//get the object being created        
		final String strUtilDate = toString(ast);
        
        //did we construct a java.util.Date object?
        if(!"java.util.Date".equals(strUtilDate) && !"Date".equals(strUtilDate)) {
            return;
        }
        
        //did we immediate call a method on that object?
        DetailAST astGetTimeMethod = getParentType(ast, TokenTypes.METHOD_CALL);
        if (astGetTimeMethod == null) {
            return;
        }
        
        //get the name of the method...
        String strGetTimeMethod = toString(astGetTimeMethod);        
        int iLastDot = strGetTimeMethod.lastIndexOf('.');
        if (iLastDot != -1) {
        	strGetTimeMethod = strGetTimeMethod.substring(iLastDot+1);
        }
        
        //was it getTime() ???
        if (!"getTime".equals(strGetTimeMethod)) {
        	return;
        }
        
        //we have an issue... they should have used System.currentTimeMillis() instead
        log(ast, "wastefulDateCreationCheck.useSystemMillis");
	}
	
}
