/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *  
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *  
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 * 
 */
package com.jimischopp.checkstyle;


import com.puppycrawl.tools.checkstyle.api.DetailAST;
import com.puppycrawl.tools.checkstyle.api.TokenTypes;


/**
 * Checks for unnecessary wrapper object creation used to parse strings and primitives.
 * This is a very novice programming mistake that nevertheless occurs. It finds code like:
 * <pre>
 *        short s = new Short("5").shortValue();
 * </pre>
 * It also subsequently finds truly creative novice code such as:
 * <pre>
 *        int i = new Integer("-" + x).intValue();
 * </pre>
 * and
 * <pre>
 *        int i = new Long(-1).intValue();
 * </pre>
 * 
 * <p>
 * Copyright 2003, James Schopp
 *
 * @author James Schopp
 * @see com.puppycrawl.tools.checkstyle.api.Check
 * @since Oct 27, 2003
 *
 *
 */
public class WastefulParseNumCheck extends AbstractBaseChecker {
       
    private final static String PKG_JAVA_LANG = "java.lang.";
    
    private final static short NOTWRAPPER   = -1;
    private final static short LONG         = 0;
    private final static short SHORT        = 1;
    private final static short INTEGER      = 2;
    private final static short DOUBLE       = 3;
    private final static short FLOAT        = 4;
    private final static short BYTE         = 5;
    private final static short STRING       = 6;
    
    private final static String[] WRAPPERTYPES = new String[] {
                                                                "Long", 
                                                                "Short", 
                                                                "Integer",
                                                                "Double",
                                                                "Float",
                                                                "Byte",
                                                                };
    
    private final static String[] WRAPPERMETHODS = new String[] {
                                                                  "longValue",
																  "shortValue",
																  "intValue",
																  "doubleValue",
																  "floatValue",
																  "byteValue",
                                                                  "toString",
                                                                };
    
        	
    public int[] getDefaultTokens()
	{
    	return new int[]{TokenTypes.LITERAL_NEW, };
	}
    
    public int[] getAcceptableTokens()
	{
    	return getDefaultTokens();
	}
        
    public int[] getRequiredTokens()
	{
    	return getDefaultTokens();
	}
    
    private short getWrapperType(String strType)
	{        
        //get rid of the leading package name if it is there...
    	if (strType.indexOf(PKG_JAVA_LANG) == 0) {
    		strType = strType.substring(PKG_JAVA_LANG.length()-1);    		
    	}
        
        //find out what wrapper type it is...
        if (WRAPPERTYPES[LONG].equals(strType)) {
            return LONG;
                    
        } else if (WRAPPERTYPES[SHORT].equals(strType)) {
            return SHORT;
            
        } else if (WRAPPERTYPES[INTEGER].equals(strType)) {
            return INTEGER;
            
        } else if (WRAPPERTYPES[DOUBLE].equals(strType)) {
            return DOUBLE;
            
        } else if (WRAPPERTYPES[FLOAT].equals(strType)) {
            return FLOAT;
            
        } else if (WRAPPERTYPES[BYTE].equals(strType)) {
            return BYTE;
           
        }
        
        return NOTWRAPPER;        
	}
    
    
    private short getWrapperMethod(String strMethod)
    {
        //get trid of the leading variable name from the method call... 
        int iLastDot = strMethod.lastIndexOf('.');
        if (iLastDot != -1) {
            strMethod = strMethod.substring(iLastDot+1);
        }
        
        //find out what wrapper method it is...
        if (WRAPPERMETHODS[LONG].equals(strMethod)) {
            return LONG;
                    
        } else if (WRAPPERMETHODS[SHORT].equals(strMethod)) {
            return SHORT;
            
        } else if (WRAPPERMETHODS[INTEGER].equals(strMethod)) {
            return INTEGER;
            
        } else if (WRAPPERMETHODS[DOUBLE].equals(strMethod)) {
            return DOUBLE;
            
        } else if (WRAPPERMETHODS[FLOAT].equals(strMethod)) {
            return FLOAT;
            
        } else if (WRAPPERMETHODS[BYTE].equals(strMethod)) {
            return BYTE;           
        
        } else if (WRAPPERMETHODS[STRING].equals(strMethod)) {
            return STRING;           
        }
                
        return NOTWRAPPER;        
    }
        
	public void visitToken(final DetailAST ast)
	{
		//get the type that is being constructed        
        final String strType = toString(ast);
        
        //if the variable is not a wrapper class, then we don't care...
        final short shtType = getWrapperType(strType);
        if (NOTWRAPPER == shtType) {        
            return;            
        }
        
        //did they immediately call a method on the new object?
        final DetailAST astWrapperMeth = getParentType(ast, TokenTypes.METHOD_CALL);
        if (null == astWrapperMeth) {
            return;
        }
        
        //get the method they called...
        String strMethod = toString(astWrapperMeth);        
                
        //was the method a wrapper method?
        final short shtMethod = getWrapperMethod(strMethod); 
        if (NOTWRAPPER == shtMethod) {
        	return;
        }
        
        log(ast, "wastefulParseNumCheck");
	}
  
}
